
rm(list=ls())
library(dplyr)
library(Synth)

dat = read.csv("../RAS.csv", header=TRUE)

dat$country = as.character(dat$country)

dat = dat[, c("country", "ctyid", "year", "polity", "gdppc",  "rdi", "muslim", 
              "state", "social")]

set.seed(12345)

# Different Types of Countries
list.mena = c("Tunisia","Egypt","Libya","Yemen","Bahrain","Syria","Algeria","Iraq","Jordan","Kuwait","Morocco","Oman","Sudan",
              "Mauritania","Saudi Arabia","Djibouti","Western Sahara","Palestinian Authority (West Bank)")
list.major = c("Bahrain","Syria","Algeria","Iraq","Jordan","Kuwait","Morocco","Oman","Sudan")
list.regime.change = c("Tunisia","Egypt","Libya","Yemen")


mena = aggregate(dat[dat$country %in% list.mena , c("polity","gdppc","rdi","state","social")],
                 by=list(year=dat[dat$country %in% list.mena , "year"]),
                 FUN = mean, na.rm=TRUE)
major = aggregate(dat[dat$country %in% list.major , c("polity","gdppc","rdi","state","social")],
                  by=list(year=dat[dat$country %in% list.major , "year"]),
                  FUN = mean, na.rm=TRUE)
regime.change = aggregate(dat[dat$country %in% list.regime.change , c("polity","gdppc","rdi","state","social")],
                          by=list(year=dat[dat$country %in% list.regime.change , "year"]),
                          FUN = mean, na.rm=TRUE)


mena = data.frame(country="Uprising Countries", ctyid=1000, muslim=1, mena)
major = data.frame(country="Major Protest Countries", ctyid=1000, muslim=1, major)
regime.change = data.frame(country="Regime Change Countries", ctyid=1000, muslim=1, regime.change)


### CHANGE HERE TO SPECIFY WHICH COUNTRIES TO COMBINE
#dat = rbind(dat, mena)
#add.name = " In-Time (MENA Countries)"
#treatment.identifier = "Uprising Countries"

#dat = rbind(dat, major)
#add.name = " In-Time (Major Protest)"
#treatment.identifier = "Major Protest Countries"

dat = rbind(dat, regime.change)
add.name = " In-Time (Regime Change)"
treatment.identifier = "Regime Change Countries"


predictors.global = c("polity","gdppc","rdi","social","state")


##############################################################
# Create not.in function to make life easier
##############################################################
`%not.in%` <- function(x,y)!('%in%'(x,y))


##############################################################
# Delete Incomplete Countries as Preparation for Synthetic Control
##############################################################

num.missing <- function(x) {
  return (sum(is.na(x)))
}

dat.temp = dat
out = aggregate(x=dat.temp, by=list(country=dat.temp$country), FUN=num.missing)

has.missing = rowSums((out[,-1])) > 0
countries.without.missing = out$country[!has.missing]

dat = dat[dat$country %in% countries.without.missing,]



##############################################################
# defining global parameters
##############################################################

# CHANGE HERE AS WELL TO SPECIFY DONOR POOL
controls.identifier = unique(dat$country[dat$country %not.in% c(treatment.identifier, list.regime.change, list.major)])



time = 1990:2010
time.pretreat = 1990:2007

# global parameters for graphic
width = 3200
height = 2400
res = 300



##############################################################
# synthetic control estimation
##############################################################

build.data <- function(dat.synth, predictors, dependent, treatment.identifier, controls.identifier, 
                       time, time.pretreat) {
  
  return(
    dataprep(
      foo = dat.synth,
      predictors = predictors,
      predictors.op = "mean",
      dependent = dependent,
      unit.variable = "ctyid",
      time.variable = "year",
      treatment.identifier = treatment.identifier,
      controls.identifier = controls.identifier,
      time.predictors.prior = time.pretreat,
      time.optimize.ssr = time.pretreat,
      unit.names.variable = "country",
      time.plot = time
    )
  )
}


# Draw Trends
draw.trends <- function(Y0, Y1, x.range, ylim, ylab, legend.lab, time.v, 
                        text.x, text.y, text) {

    plot(x.range,Y1,
       type="l", ylim=ylim, 
       col=rgb(0,0,0,alpha=.3), lty="solid",
       ylab = ylab,
       xlab ="Year",
       lwd=6,
       axes = "false",
       cex.lab=1.5
  )
  axis(side=1, at=seq(time[1], time[length(time)], 2))
  axis(side=2, at=seq(ylim[1], ylim[2], 5))
  
  lines(x.range,Y0,col="black",lty="dashed",lwd=2)
  abline(v=time.v-1, lty="dotted")
  abline(v=time.v, lty="dotted")
  text(text.x-.5, text.y, text, cex=1.4)
  legend(x="bottomright",
         legend=legend.lab,
         lty=c("solid","dashed"),col=c("black","black"),
         cex=1.4,bg="white",lwd=c(2,2))
  
}



# Draw Gap
draw.gap <- function(Y0, Y1, x.range, ylim, ylab, time.v, text.x, text.y, text) {
  plot(x.range, Y1 - Y0,
       type="l", ylim=ylim, 
       col=rgb(0,0,0,alpha=.3), lty="solid",
       ylab = ylab,
       xlab ="Year",
       lwd=6,
       axes = FALSE,
       cex.lab = 1.5
  )
  axis(side=1, at=seq(time[1], time[length(time)], 2))
  axis(side=2, at=seq(ylim[1], ylim[2], 5))
  
  abline(v=time.v-1, lty="dotted")
  abline(v=time.v, lty="dotted")
  text(text.x-.5, text.y, text, cex=1.4)
  abline(h=0, lty="dotted")
}



##############################################################
# state
##############################################################

dependent = "state"
predictors = unique(c(predictors.global, dependent))


dataprep.out <- build.data(dat.synth = dat,
                           predictors = predictors, 
                           dependent = dependent, 
                           treatment.identifier, 
                           controls.identifier,
                           time,
                           time.pretreat)

synth.out <- synth(dataprep.out)


png(filename = paste0(treatment.identifier," Synthetic Government", add.name, ".png"), width=width, height=height, res=res)
draw.trends(Y0 = (dataprep.out$Y0%*%synth.out$solution.w),
            Y1 = dataprep.out$Y1plot,
            x.range = time,
            ylim = c(0, 65),
            ylab = "Government Discrimination of Religious Minorities",
            legend.lab = c(treatment.identifier, paste("Synthetic",treatment.identifier)),
            time.v = 2008,
            text.x = 2008, text.y = 10, text = "Hypothetical\nArab Uprisings"
            )
dev.off()

png(filename = paste0(treatment.identifier," Gap Government", add.name, ".png"), width=width, height=height, res=res)
draw.gap(Y0 = (dataprep.out$Y0%*%synth.out$solution.w),
            Y1 = dataprep.out$Y1plot,
            x.range = time,
            ylim = c(-30, 30),
            ylab = "Gap in Government Discrimination (Actual - Synthetic)",
            time.v = 2008,
            text.x = 2008, text.y = -5, text = "Hypothetical\nArab Uprisings"
)
dev.off()


synth.tables <- synth.tab(
  dataprep.res = dataprep.out,
  synth.res = synth.out)
sink(file = paste0("Synthetic State", add.name, ".txt"))
print(synth.tables)
sink(NULL)


##############################################################
# social
##############################################################

dependent = "social"
predictors = unique(c(predictors.global, dependent))


dataprep.out <- build.data(dat.synth = dat,
                           predictors = predictors, 
                           dependent = dependent, 
                           treatment.identifier, 
                           controls.identifier,
                           time,
                           time.pretreat)

synth.out <- synth(dataprep.out)

png(filename = paste0(treatment.identifier," Synthetic Social", add.name, ".png"), width=width, height=height, res=res)
draw.trends(Y0 = (dataprep.out$Y0%*%synth.out$solution.w),
            Y1 = dataprep.out$Y1plot,
            x.range = time,
            ylim = c(0, 65),
            ylab = "Social Discrimination of Religious Minorities",
            legend.lab = c(treatment.identifier, paste("Synthetic",treatment.identifier)),
            time.v = 2008,
            text.x = 2008, text.y = 10, text = "Hypothetical\nArab Uprisings"
)
dev.off()

png(filename = paste0(treatment.identifier," Gap Social", add.name, ".png"), width=width, height=height, res=res)
draw.gap(Y0 = (dataprep.out$Y0%*%synth.out$solution.w),
         Y1 = dataprep.out$Y1plot,
         x.range = time,
         ylim = c(-30, 30),
         ylab = "Gap in Social Discrimination (Actual - Synthetic)",
         time.v = 2008,
         text.x = 2008, text.y = -5, text = "Hypothetical\nArab Uprisings"
)
dev.off()

synth.tables <- synth.tab(
  dataprep.res = dataprep.out,
  synth.res = synth.out)
sink(file = paste0("Synthetic Social", add.name, ".txt"))
print(synth.tables)
sink(NULL)

